--[[
Interface: 1.7.1.0 b10490

Copyright (C) GtX (Andy), 2018

Author: GtX | Andy
Date: 08.10.2018

History:
V 1.0.1.0 @ 29.03-2021 - Fixed controller conflict on Main Map screen when trying to purchase lands.

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No changes are to be made to this script without permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
An diesem Skript dürfen ohne Genehmigung von GtX | Andy keine Änderungen vorgenommen werden
]]


local universalPassenger
local loadError

local versionString = "0.0.0.0"
local buildId = 2

local modDirectory = g_currentModDirectory
local modName = g_currentModName

local function isActive()
    return universalPassenger ~= nil
end

local function registerActionEvents(mission)
    if isActive() then
        universalPassenger:registerActionEvents()
    end
end

local function unregisterActionEvents(mission)
    if isActive() then
        universalPassenger:unregisterActionEvents()
    end
end

local function onConnectionFinishedLoading(mission, connection, x, y, z, viewDistanceCoeff)
    if mission:getIsServer() and isActive() then
        universalPassenger:onConnectionFinishedLoading(connection)
    end
end

local function onConnectionClosed(mission, connection)
    if mission:getIsServer() and isActive() then
        for _, passengerVehicle in pairs(universalPassenger.passengerVehicles) do
            local spec = passengerVehicle.spec_universalPassenger

            if spec ~= nil and spec.connectionToSeatId[connection] ~= nil then
                local seatId = spec.connectionToSeatId[connection]
                local seat = spec.passengerSeats[seatId]

                universalPassenger:logPrint(5, "Kicking %s from seat id %d!", seat.passengerName, seatId)

                spec.connectionToSeatId[connection] = nil
                passengerVehicle:exitPassengerSeat(seatId, UniversalPassengerSpec.EXIT)
                g_server:broadcastEvent(UniversalPassengerExitEvent:new(passengerVehicle, seatId, UniversalPassengerSpec.EXIT))
            end
        end
    end
end

local function requestToEnterVehicle(mission, vehicleToEnter)
    if isActive() then
        local vehicle, seatId = universalPassenger:getVehicleAndSeatId()

        if vehicle ~= nil then
            if mission:getIsServer() then
                vehicle:exitPassengerSeat(seatId, UniversalPassengerSpec.ENTER_DRIVER)
                g_server:broadcastEvent(UniversalPassengerExitEvent:new(vehicle, seatId, UniversalPassengerSpec.ENTER_DRIVER))
            else
                g_client:getServerConnection():sendEvent(UniversalPassengerExitEvent:new(vehicle, seatId, UniversalPassengerSpec.ENTER_DRIVER))
                vehicle:exitPassengerSeat(seatId, UniversalPassengerSpec.ENTER_DRIVER)
            end
        end
    end
end

local function getPositionData(player, superFunc)
    if isActive() and universalPassenger.playerToVehicle[player] ~= nil then
        local vehicle = universalPassenger.playerToVehicle[player]

        local posX, posY, posZ = getTranslation(vehicle.rootNode)
        local dx, _, dz = localDirectionToWorld(vehicle.rootNode, 0, 0, 1)
        local yRot = 0

        if vehicle.spec_drivable ~= nil and vehicle.spec_drivable.reverserDirection == -1 then
            yRot = MathUtil.getYRotationFromDirection(dx, dz)
        else
            yRot = MathUtil.getYRotationFromDirection(dx, dz) + math.pi
        end

        return posX, posY, posZ, yRot
    end

    return superFunc(player)
end

local function drawPassengerNames(ingameMap, showNames, leftBorderReached, rightBorderReached, topBorderReached, bottomBorderReached)
    if g_currentMission.missionDynamicInfo.isMultiplayer and isActive() then
        local resetOffsetMultiplier = true
        local mapArrow = ingameMap.mapArrowRedOverlay

        for player, vehicle in pairs(universalPassenger.playerToVehicle) do
            if player ~= g_currentMission.player and player.visualInformation ~= nil then
                local spec = vehicle.spec_universalPassenger

                if resetOffsetMultiplier or spec.playerNameOffsetMultiplier == nil then
                    spec.playerNameOffsetMultiplier = 1
                    resetOffsetMultiplier = false
                end

                local posX, rotY, posZ = getTranslation(vehicle.rootNode)
                local dx, _, dz = localDirectionToWorld(vehicle.rootNode, 0, 0, 1)

                posX = (math.floor(posX) + ingameMap.worldCenterOffsetX) / ingameMap.worldSizeX
                posZ = (math.floor(posZ) + ingameMap.worldCenterOffsetZ) / ingameMap.worldSizeZ

                if ingameMap:setMapObjectOverlayPosition(mapArrow, posX, posZ, mapArrow.width, mapArrow.height, true, showNames, leftBorderReached, rightBorderReached, topBorderReached, bottomBorderReached) then
                    -- Only draw the arrow if there is no driver so there is not a mess of arrows
                    if not vehicle:getIsControlled() and vehicle ~= universalPassenger.currentPassengerVehicle then
                        local r, g, b, a = ingameMap:colorForFarm(vehicle:getActiveFarm())

                        if vehicle.spec_drivable ~= nil and vehicle.spec_drivable.reverserDirection == -1 then
                            rotY = MathUtil.getYRotationFromDirection(dx, dz)
                        else
                            rotY = MathUtil.getYRotationFromDirection(dx, dz) + math.pi
                        end

                        ingameMap:setMapObjectOverlayRotation(mapArrow, rotY)
                        mapArrow:setColor(r, g, b, a)
                        mapArrow:render()
                    end

                    -- Not for miniMap or when zoomed out a long way
                    if showNames then
                        local text = tostring(player.visualInformation.playerName)
                        local textWidth = getTextWidth(ingameMap.playerFontSize, text)
                        local textHeight = getTextHeight(ingameMap.playerFontSize, text)
                        -- local r, g, b, a = ingameMap:colorForFarm(player.farmId)

                        setTextBold(false)
                        setTextAlignment(RenderText.ALIGN_LEFT)
                        setTextColor(1, 1, 1, 1)

                        posX = MathUtil.clamp(mapArrow.x + mapArrow.width * 0.5 + ingameMap.playerNameOffsetX - textWidth * 0.5, ingameMap.mapPosX, ingameMap.mapPosX + ingameMap.mapWidth - textWidth)
                        renderText(posX, mapArrow.y + ingameMap.playerNameOffsetY - (textHeight + ingameMap.playerNameOffsetY) * spec.playerNameOffsetMultiplier, ingameMap.playerFontSize, text)

                        spec.playerNameOffsetMultiplier = spec.playerNameOffsetMultiplier + 1
                    end
                end
            end
        end
    end
end

local function getConfigurationsFromXML(xmlFile, superFunc, baseXMLName, baseDir, customEnvironment, isMod, storeItem)
    local vehicleConfigurations = superFunc(xmlFile, baseXMLName, baseDir, customEnvironment, isMod, storeItem)

    if isActive() and baseXMLName == "vehicle" and vehicleConfigurations ~= nil then
        -- This is only for mods with Universal Passenger positions pre-added.
        if hasXMLProperty(xmlFile, "vehicle.universalPassenger") and vehicleConfigurations["universalPassenger"] == nil then
            vehicleConfigurations["universalPassenger"] = {
                {desc = "", price = 0, dailyUpkeep = 0, isDefault = true, index = 1, name = universalPassenger.texts.configurationOne},
                {desc = "", price = 0, dailyUpkeep = 0, isDefault = false, index = 2, name = universalPassenger.texts.configurationTwo},
                {desc = "", price = 0, dailyUpkeep = 0, isDefault = false, index = 3, name = universalPassenger.texts.configurationThree}
            }
        end
    end

    return vehicleConfigurations
end

local function finalizeVehicleTypes(vehicleTypeManager)
    if isActive() then
        local numInserted = 0

        local specializationName = string.format("%s.universalPassenger", modName)
        local specializationObject = g_specializationManager:getSpecializationObjectByName(specializationName)

        if specializationObject ~= nil then
            for typeName, typeEntry in pairs(vehicleTypeManager.vehicleTypes) do
                local canInsert = specializationObject.prerequisitesPresent(typeEntry.specializations)

                if canInsert then
                    for specName, _ in pairs(typeEntry.specializationsByName) do
                        if string.find(specName:upper(), "UNIVERSALPASSENGER") then
                            canInsert = false
                            break
                        end
                    end

                    if canInsert then
                        g_vehicleTypeManager:addSpecialization(typeName, specializationName)
                        numInserted = numInserted + 1
                    end
                end
            end
        end

        if numInserted == 0 then
            print("  Error: [UniversalPassenger] - Failed to insert specialization into any vehicle types!")
        end
    end
end

local function finalizeDelete(mission)
    if isActive() then
        removeModEventListener(universalPassenger)
        -- g_messageCenter:unsubscribeAll(universalPassenger)

        universalPassenger:delete()

        getfenv(0)["g_universalPassenger"] = nil
        universalPassenger = nil
    end
end

local function inGameMapOnLoadMapFinished(frame)
    if isActive() then
        universalPassenger.gameLoaded = true
        universalPassenger.pageMapOverview = frame
        universalPassenger.alpineDLCLoaded = g_modIsLoaded["pdlc_alpineFarmingPack"]

        local passengerElement = frame.buttonEnterVehicle:clone(frame)

        passengerElement:setText(universalPassenger.texts.enterPassengerSeat)
        passengerElement:setInputAction("TOGGLE_STORE") -- Using this to avoid controller conflicts

        function onClickCallback(frame)
            if frame.currentHotspot ~= nil and frame.currentHotspot.objectId ~= nil then
                local vehicle = g_currentMission.nodeToObject[frame.currentHotspot.objectId]

                if universalPassenger ~= nil and universalPassenger:getCanEnterAsPassenger(vehicle) and not vehicle:getIsPassenger() then
                    frame.onClickBackCallback()
                    universalPassenger:requestToEnterVehicle(vehicle)
                end

                return false
            end

            return true
        end

        passengerElement.onClickCallback = onClickCallback
        frame.onClickEnterPassengerVehicle = onClickCallback

        frame.buttonEnterVehicle.parent:addElement(passengerElement)
        frame.buttonEnterAsPassenger = passengerElement

        frame.setMapSelectionItem = Utils.overwrittenFunction(frame.setMapSelectionItem, function (frame, superFunc, hotspot)
            local hasPassengerButton = frame.buttonEnterAsPassenger ~= nil

            if hasPassengerButton and frame.buttonEnterAsPassenger:getIsVisible() then
                frame.buttonEnterAsPassenger:setVisible(false)
            end

            frame.canEnterPassengerSeat = false

            superFunc(frame, hotspot)

            if hasPassengerButton then
                if frame.currentHotspot ~= nil and frame.currentHotspot.objectId ~= nil then
                    local vehicle = g_currentMission.nodeToObject[frame.currentHotspot.objectId]

                    if isActive() and universalPassenger:getCanEnterAsPassenger(vehicle) and not vehicle:getIsPassenger() then
                        frame.canEnterPassengerSeat = true

                        if not frame.buttonBox:getIsVisible() then
                            frame.buttonBox:setVisible(isVisible)
                        end

                        frame.buttonEnterAsPassenger:setVisible(true)
                        frame.buttonLayout:invalidateLayout()
                    end
                end
            end
        end)

        frame.registerInput = Utils.appendedFunction(frame.registerInput, function (frame)
            if frame.onClickEnterPassengerVehicle ~= nil then
                local eventId, state = frame.inputManager:registerActionEvent(InputAction.TOGGLE_STORE, frame, frame.onClickEnterPassengerVehicle, false, true, false, true)
            end
        end)

        frame.unregisterInput = Utils.appendedFunction(frame.unregisterInput, function (frame)
            if frame.buttonEnterAsPassenger ~= nil then
                frame.inputManager:removeActionEventsByActionName(InputAction.TOGGLE_STORE)
            end
        end)

        frame.onClickVisitPlace = Utils.overwrittenFunction(frame.onClickVisitPlace, function (frame, superFunc)
            if isActive() and universalPassenger:getIsPassenger() then
                if frame.currentHotspot ~= nil and (frame.currentHotspot.xMapPos ~= nil and frame.currentHotspot.zMapPos ~= nil) then
                    frame.onClickBackCallback()
                    universalPassenger:exitVehicles(frame.currentHotspot.xMapPos, 1.5, frame.currentHotspot.zMapPos, false, false)
                end
            else
                superFunc(frame)
            end
        end)

        frame.buttonVisitPlace.onClickCallback = Utils.overwrittenFunction(frame.buttonVisitPlace.onClickCallback, function (frame, superFunc)
            if isActive() and universalPassenger:getIsPassenger() then
                if frame.currentHotspot ~= nil and (frame.currentHotspot.xMapPos ~= nil and frame.currentHotspot.zMapPos ~= nil) then
                    frame.onClickBackCallback()
                    universalPassenger:exitVehicles(frame.currentHotspot.xMapPos, 1.5, frame.currentHotspot.zMapPos, false, false)
                end
            else
                superFunc(frame)
            end
        end)
    end
end

local function init()
    if g_universalPassenger ~= nil then
        g_universalPassenger:logPrint(3, "Trying to load mod twice! Only '1' copy is required in your mod folder. Please remove '%s'", modName)
    end

    local mod = g_modManager:getModByName(modName)
    versionString = tostring(mod.version) or versionString

    if mod.modName == "FS19_UniversalPassenger" or mod.modName == "FS19_UniversalPassenger_update" and mod.author:len() == 3 then
        local xmlFilename = modDirectory .. "xml/BaseVehicles.xml"

        if fileExists(xmlFilename) then
            source(modDirectory .. "scripts/UniversalPassenger.lua")

            source(modDirectory .. "scripts/events/UniversalPassengerExitEvent.lua")
            source(modDirectory .. "scripts/events/UniversalPassengerEnterEvent.lua")
            source(modDirectory .. "scripts/events/UniversalPassengerNextSeatEvent.lua")
            source(modDirectory .. "scripts/events/UniversalPassengerRequestEnterEvent.lua")

            universalPassenger = UniversalPassenger:new(g_server ~= nil, g_dedicatedServerInfo == nil, modName, modDirectory, xmlFilename, versionString, buildId)

            if universalPassenger ~= nil then
                addModEventListener(universalPassenger)

                getfenv(0)["g_universalPassenger"] = universalPassenger

                g_specializationManager:addSpecialization("universalPassenger", "UniversalPassengerSpec", modDirectory .. "scripts/spec/UniversalPassengerSpec.lua")

                if universalPassenger.isServer and universalPassenger.isClient then
                    source(modDirectory .. "scripts/misc/UP_Creator.lua")
                end

                FSBaseMission.registerActionEvents = Utils.appendedFunction(FSBaseMission.registerActionEvents, registerActionEvents)
                BaseMission.unregisterActionEvents = Utils.appendedFunction(BaseMission.unregisterActionEvents, unregisterActionEvents)

                FSBaseMission.onConnectionFinishedLoading = Utils.appendedFunction(FSBaseMission.onConnectionFinishedLoading, onConnectionFinishedLoading)
                FSBaseMission.onConnectionClosed = Utils.prependedFunction(FSBaseMission.onConnectionClosed, onConnectionClosed)

                BaseMission.requestToEnterVehicle = Utils.prependedFunction(BaseMission.requestToEnterVehicle, requestToEnterVehicle)

                Player.getPositionData = Utils.overwrittenFunction(Player.getPositionData, getPositionData)
                IngameMap.drawEnterableArrows = Utils.appendedFunction(IngameMap.drawEnterableArrows, drawPassengerNames)

                StoreItemUtil.getConfigurationsFromXML = Utils.overwrittenFunction(StoreItemUtil.getConfigurationsFromXML, getConfigurationsFromXML)

                VehicleTypeManager.finalizeVehicleTypes = Utils.prependedFunction(VehicleTypeManager.finalizeVehicleTypes, finalizeVehicleTypes)
                FSBaseMission.delete = Utils.appendedFunction(FSBaseMission.delete, finalizeDelete)

                InGameMenuMapFrame.onLoadMapFinished = Utils.appendedFunction(InGameMenuMapFrame.onLoadMapFinished, inGameMapOnLoadMapFinished)

                print("  Loaded Universal Passenger")
            end
        end
    else
        loadError = {
            author = mod.author,
            startUpdateTime = 2000,

            update = function(self, dt)
                self.startUpdateTime = self.startUpdateTime - dt

                if self.startUpdateTime < 0 then
                    local title = string.format("Universal Passenger - Version %s", versionString)
                    local text = string.format(g_i18n:getText("universalPassenger_loadError", modName), modName, self.author)

                    if g_dedicatedServerInfo == nil then
                        if not g_gui:getIsGuiVisible() then
                            g_gui:showYesNoDialog({
                                title = title,
                                text = text,
                                dialogType = DialogElement.TYPE_LOADING,
                                callback = self.openModHubLink,
                                target = nil,
                                yesText = g_i18n:getText("button_ok"),
                                noText = g_i18n:getText("button_modHubDownload")
                            })
                        end
                    else
                        print("\n" .. text .. "\n    - https://farming-simulator.com/mods.php?lang=en&country=be&title=fs2019&filter=org&org_id=129652&page=0" .. "\n")
                        self.openModHubLink(true)
                    end
                end
            end,

            openModHubLink = function(ignore)
                if ignore == false then
                    local language = g_languageShort
                    local link = "mods.php?lang=en&country=be&title=fs2019&filter=org&org_id=129652&page=0"
                    if language == "de" or language == "fr" then
                        link = "mods.php?lang=" .. language .. "&country=be&title=fs2019&filter=org&org_id=129652&page=0"
                    end

                    openWebFile(link, "")
                end

                removeModEventListener(loadError)
                loadError = nil
            end
        }

        addModEventListener(loadError)
    end
end

init()
