--[[

Game:	Farming Simulator 19
Title:	HarvestMaster Utility Script
Author:	ThundR

--]]

-- [Decalrations] ***********************************************************************************************************************************

-- Switch to global ClassUtil
ClassUtil = getfenv(0).ClassUtil

HMUtils = {}

-- Message constants
HMUtils.MSG = {
	INTERNAL_ERROR				= "An internal error has occurred",
	FILE_NOT_FOUND				= "Could not find file\n%s", -- includes filename
	INVALID_XML_FILE			= "Invalid xml file \n%s", -- includes filename
	INVALID_XML_VALUE	 		= "Invalid xml value for %s\n%s", -- includes value and filename
	INVALID_XML_VALUE_AT_INDEX	= "Invalid xml value for %s at index (%s)\n%s", -- includes value, index, and filename
	FAILED_TO_CEATE_XML_FILE	= "Failed to create xml file\n%s", -- includes filename
	INVALID_ARGUMENT			= "Invalid argument %q",
	INVALID_VALUE				= "Invalid %s", -- This could be used for any invalid message
	NO_RETURN_VALUE				= "No return value",
	INDEX_OUT_OF_RANGE			= "Index out of range",
	INVALID_TABLE				= "Invalid table",
	INVALID_TABLE_NAME			= "Invalid table (%s)",
	FUNCTION_EXISTS				= "Function %q already exists",
	INVALID_FUNCTION			= "Invalid function",
	INVALID_FUNCTION_CALL		= "Invalid function call",
	INVALID_FUNCTION_NAME		= "Invalid function (%s)",
	INVALID_OLD_FUNCTION		= "Invalid original function",
	INVALID_OLD_FUNCTION_NAME	= "Invalid original function (%s)",
	INVALID_NEW_FUNCTION		= "Invalid target function",
	INVALID_NEW_FUNCTION_NAME	= "Invalid target function (%s)",
	INVALID_SAVED_FUNCTION		= "Invalid saved function (%s)"
}

-- Math Constants
HMUtils.MATH = {
	ROUND = "ROUND",
	FLOOR = "FLOOR",
	CEIL  = "CEIL"
}

-- Debug Constants
HMUtils.DEBUG = {
	ON		= 0,
	OFF		= -math.huge,
	FULL	= math.huge,
	ERROR	= -1000000,
	TESTING = 1000000
}

-- Debug Variables
HMUtils.globalDebugLevel = HMUtils.DEBUG.ON
HMUtils.globalDebug = HMUtils.globalDebugLevel > HMUtils.DEBUG.OFF

-- [General Utilities] ******************************************************************************************************************************

function HMUtils.isBoolean(arg)
-- Returns true if value is boolean
	-- [arg]: Value to check

	if type(arg) == "boolean" then
		return true
	end
end

function HMUtils.isString(arg)
-- Returns true if value is a string
	-- [arg]: Value to check

	if type(arg) == "string" then
		return true
	end
end

function HMUtils.isNumber(arg)
-- Returns true if value is a number
	-- [arg]: Value to check

	if tonumber(arg) then
		return true
	end
end

function HMUtils.isTable(arg)
-- Returns true if value is a table
	-- [arg]: Value to check

	if type(arg) == "table" then
		return true
	end
end

function HMUtils.isFunction(arg)
-- Returns true if value is a function
	-- [arg]: Value to check

	if type(arg) == "function" then
		return true
	end
end

function HMUtils.isApp(arg)
-- Returns true if value is an application

	if HMUtils.isTable(arg) and arg.class and arg:class() == InfoMenu then
		return true
	end
end

function HMUtils.isBetween(num, min, max, useMinMax)
-- Returns true if value is between min and max values
	-- [num]: Number to check
	-- [min]: Minimum value
	-- [max]: Maximum value
	-- [useMinMax]: Include min and max values in range > Default = true

	useMinMax = Utils.getNoNil(useMinMax, true)

	if HMUtils.argIsInvalid(HMUtils.isNumber(num), "num")
	or HMUtils.argIsInvalid(HMUtils.isNumber(min), "min")
	or HMUtils.argIsInvalid(HMUtils.isNumber(max), "max")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(useMinMax), "useMinMax")
	then
		return
	end

	if (useMinMax == true and (num == min or num == max))
	or (num > min and num < max)
	then
		return true
	end
end

-- [Message Utilities] ******************************************************************************************************************************

function HMUtils.getMsg(msg)
-- Returns a saved log message
	-- [msg]: Message to retrieve

	if not HMUtils.isString(msg) then
		return
	end

	msg = HMUtils.upperCase(msg)
	return HMUtils.MSG[msg]
end

function HMUtils.displayMsg(text, ...)
-- Displays a custom message in the log
	-- [text]:	Text to display
	-- [...]:	Variables to pass to the built-in string.format function > Optional

	text = Utils.getNoNil(HMUtils.getMsg(text), text)
	if HMUtils.argIsInvalid(HMUtils.isString(text), "text") then
		return
	end

	local title = ""
	if HMUtils.isTable(g_modDesc) and HMUtils.isString(g_modDesc.title) then
		title = string.format("[%s] ", g_modDesc.title)
	end

	print(string.format(">> %s%s", title, string.format(text, ...)))

	return true
end

function HMUtils.errorMsg(text, ...)
-- Displays an error message, showing the callstack
	-- [text]:	Text to display on error (string) > Default = INTERNAL_ERROR
	-- [...]:	Variables to pass to the built-in string.format function > Optional

	text = Utils.getNoNil(text, "INTERNAL_ERROR")

	if HMUtils.displayMsg(text, ...) then
		printCallstack()
	end
end

function HMUtils.debugMsg(...)
-- Conditionally displays a custom message in the log
-- This function behaves differently depending on how it is called
	-- [app]:	Source application > Optional
	-- [lvl]:	Debug level of message > Optional
	-- [text]:	Message to display

	local args = HMUtils.pack(...)
	local app = args[1]
	local lvl = args[2]
	local txt = args[3]
	local var = 4

	if not HMUtils.isApp(app) then
		if HMUtils.isNumber(app) then
			app = nil
			lvl = args[1]
			txt = args[2]
			var = 3
		elseif HMUtils.isString(app) then
			app = nil
			lvl = nil
			txt = args[1]
			var = 2
		else
			app = nil
			lvl = nil
			txt = nil
			var = nil
		end
	end

	if HMUtils.errorOnFalse(var, "INVALID_FUNCTION_CALL")
	or var >= 4 and HMUtils.argIsInvalid(HMUtils.isApp(app), "app")
	or var >= 3 and HMUtils.argIsInvalid(HMUtils.isNumber(lvl), "lvl")
	then
		return
	end

	txt = Utils.getNoNil(HMUtils.getMsg(txt), txt)
	if HMUtils.argIsInvalid(HMUtils.isString(txt), "txt") then
		return
	end
	txt = "<DEBUG>: "..txt

	local showDebug = false
	if (app and lvl and lvl <= app.debugLevel)
	or (lvl and lvl <= HMUtils.globalDebugLevel)
	or (lvl == nil and HMUtils.globablDebug == true)
	then
		showDebug = true
	end

	if showDebug == true then
		return HMUtils.displayMsg(txt, HMUtils.unpack(..., var))
	end
end

function HMUtils.assert(expression, text, ...)
-- Returns the result of an expression, displaying an error on nil or false
	-- [expression]:	Expression to check
	-- [text]:			Text to display on error
	-- [...]:			Variables to pass to the built-in string.format function > Optional

	if not expression then
		HMUtils.errorMsg(text, ...)
	end

	return expression
end

function HMUtils.errorOnTrue(expression, text, ...)
-- Returns true and displays an error if the result of the passed expression is not nil or false
-- Shortcut for "already exists" checks
	-- [expression]: Expression to evaluate
	-- [text]:		 Text to display on error
	-- [...]:		 Variables to pass to built-in string.format function > Optional

	if expression then
		HMUtils.errorMsg(text, ...)
		return true
	end
end

function HMUtils.errorOnFalse(expression, text, ...)
-- Returns true and displays an error if the result of the passed expression is nil or false
-- Shortcut for "invalid value" checks
	-- [expression]: Expression to evaluate
	-- [text]:		 Text to display on error
	-- [...]:		 Variables to pass to built-in string.format function > Optional

	if not expression then
		HMUtils.errorMsg(text, ...)
		return true
	end
end

-- [String Utilities] *******************************************************************************************************************************

function HMUtils.getString(val, ignoreNil, ignoreNumbers)
-- Converts a value to a string, optionally skipping nil and numeric values
-- Handles some repeat code used in various functions
	-- [val]:	 		 Value to convert
	-- [ignoreNil]: 	 Return nil values as nil instead of "nil" > Default = true
	-- [ignoreNumbers]:  Return numeric values as is instead of convering them > Default = true

	ignoreNil = Utils.getNoNil(ignoreNil, true)
	ignoreNumbers = Utils.getNoNil(ignoreNumbers, true)

	if HMUtils.argIsInvalid(HMUtils.isBoolean(ignoreNil), "ignoreNil")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(ignoreNumbers), "ignoreNumbers")
	then
		return
	end

	if not HMUtils.isString(val) then
		if (val == nil and ignoreNil == true)
		or (HMUtils.isNumber(val) and ignoreNumbers == true)
		then
			return val
		end
	end

	return tostring(val)
end

function HMUtils.upperCase(str, ignoreNil, ignoreNumbers)
-- Returns an UPPERCASE formatted string
	-- [str]:			String to format
	-- [ignoreNil]: 	 Return nil values as nil instead of "NIL" > See HMUtils.getString()
	-- [ignoreNumbers]:  Return numeric values as is instead of convering them > See HMUtils.getString()

	-- Errors handled by HMUtils.getString()

	str = HMUtils.getString(str, ignoreNil, ignoreNumbers)
	if not HMUtils.isString(str) then
		return str
	end

	return string.upper(str)
end

function HMUtils.lowerCase(str, ignoreNil, ignoreNumbers)
-- Returns a lowercase formatted string
	-- [str]:			String to format
	-- [ignoreNil]:		Return nil values as they are, instead of "nil" > See HMUtils.getString()
	-- [ignoreNumbers]: Return numeric values as they are (no conversion) > See HMUtils.getString()

	-- Errors handled by HMUtils.getString()

	str = HMUtils.getString(str, ignoreNil, ignoreNumbers)
	if not HMUtils.isString(str) then
		return str
	end

	return string.lower(str)
end

function HMUtils.camelCase(str, ignoreNil, ignoreNumbers)
-- Returns a camelCase formatted string
	-- [str]:			String to format
	-- [ignoreNil]:		Return nil values as they are, instead of "nIL" > See HMUtils.getString()
	-- [ignoreNumbers]: Return numeric values as they are (no conversion) > See HMUtils.getString()

	-- Errors handled by HMUtils.getString()

	str = HMUtils.getString(str, ignoreNil, ignoreNumbers)
	if not HMUtils.isString(str) then
		return str
	end

	local entries = HMUtils.splitString(str)
	for i, entry in ipairs(entries) do
		if i == 1 then
			entry = HMUtils.lowerCase(string.sub(entry, 1, 1))..string.sub(entry, 2)
			str = entry
		else
			entry = HMUtils.upperCase(string.sub(entry, 1, 1))..string.sub(entry, 2)
			str = str..entry
		end
	end

	return str
end

function HMUtils.properCase(str, compress, ignoreNil, ignoreNumbers)
-- Returns a ProperCase formatted string
	-- [str]:			String to format
	-- [compress]:		Remove spaces and compress into a single word (boolean) > Default = false
	-- [ignoreNil]:		Return nil values as they are, instead of "Nil" > See HMUtils.getString()
	-- [ignoreNumbers]: Return numeric values as they are (no conversion) > See HMUtils.getString()

	compress = Utils.getNoNil(compress, false)

	if HMUtils.argIsInvalid(HMUtils.isBoolean(compress), "compress")
	-- ignoreNil handled by HMUtils.getString()
	-- ignoreNumbers handled by HMUtils.getString()
	then
		return
	end

	str = HMUtils.getString(str, ignoreNil, ignoreNumbers)
	if not HMUtils.isString(str) then
		return str
	end

	local entries = HMUtils.splitString(str)
	for i, entry in ipairs(entries) do
		entry = HMUtils.upperCase(string.sub(entry, 1, 1))..string.sub(entry, 2)
		if i == 1 then
			str = entry
		elseif compress == true then
			str = str..entry
		else
			str = str.." "..entry
		end
	end

	return str
end

function HMUtils.strToNum(str)
-- Returns the numeric portion of a string or nil if there is none
	-- [str]: String to convert

	if HMUtils.isNumber(str) then
		return str
	elseif HMUtils.argIsInvalid(HMUtils.isString(str), "str") then
		return
	end

	str = string.gsub(str, ",", "") -- Remove any commas
	str = string.match(str, "%-?%d+%.?%d*")

	return tonumber(str)
end

function HMUtils.numToStr(num, precision, roundType, isVariable, showCommas)
-- Converts a number to a formatted string
	-- [num]:           Numeric value to convert
	-- [precision]:     Static or variable length precision value > Default = 0
	-- [roundType]:     Type of truncation used for fomatting > Default = "CEIL" (works best in most areas)
	-- [isVariable]:	Precision is variable if true > Default = false
	-- [showCommas]:	Group digits with commas if true > Default = false

	precision  = Utils.getNoNil(precision, 0)
	roundType  = Utils.getNoNil(roundType, "CEIL")
	isVariable = Utils.getNoNil(isVariable, false)
	showCommas = Utils.getNoNil(showCommas, false)

	if HMUtils.isString(num) then
		num = tonumber(num)
	end

	roundType = HMUtils.upperCase(roundType)

	if HMUtils.argIsInvalid(HMUtils.isNumber(num), "num")
	or HMUtils.argIsInvalid(HMUtils.isNumber(precision), "precision")
	or HMUtils.argIsInvalid(HMUtils.isString(roundType) and HMUtils.MATH[roundType], "roundType")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(isVariable), "isVariable")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(showCommas), "showCommas")
	then
		return
	end

	if isVariable == true then
		precision = precision - string.len(math.floor(num))
		precision = math.max(0, precision)
	end

	local factor = 10 ^ precision

	if roundType == HMUtils.MATH.ROUND then
		num = math.floor((num * factor) + 0.5) / factor
	elseif roundType == HMUtils.MATH.FLOOR then
		num = math.floor(num * factor) / factor
	elseif roundType == HMUtils.MATH.CEIL then
		num = math.ceil(num * factor) / factor
	end

	local str = string.format("%."..tostring(precision).."f", num)

	if showCommas == true then
		local v = 0
		while true do
			str, v = string.gsub(str, "^(%-?%d+)(%d%d%d)", "%1,%2")
			if v == 0 then
				break
			end
		end
	end

	return str
end

function HMUtils.splitString(text, pattern, convert)
-- Converts a string into a table of values
	-- [text]:    	The string of values to split
	-- [pattern]: 	The string pattern used to separate values > Default = " "
	-- [convert]:	Converts numeric values to actual numbers when true > Default = false

	pattern = Utils.getNoNil(pattern, " ")
	convert = Utils.getNoNil(convert, false)

	if HMUtils.argIsInvalid(HMUtils.isString(text), "text")
	or HMUtils.argIsInvalid(HMUtils.isString(pattern), "pattern")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(convert), "convert")
	then
		return
	end

	local retTbl = StringUtil.splitString(pattern, text)

	if convert == true then
		for i = 1, #retTbl do
			retTbl[i] = Utils.getNoNil(tonumber(retTbl[i]), retTbl[i])
		end
	end

	return retTbl
end

function HMUtils.createItemList(items)
-- Creates or formats an accepted items list
	-- [items]:	Items to add

	if HMUtils.argIsInvalid(HMUtils.isTable(items) or HMUtils.isNumber(items) or HMUtils.isString(items), "items") then
		return
	end

	local list = {}

	if HMUtils.isNumber(items) then
		list[items] = true
	else
		if not HMUtils.isTable(items) then
			items = HMUtils.splitString(items, " ", true)
		end

		for key, val in pairs(items) do
			if val == true then
				list[key] = val
			else
				list[val] = true
			end
		end
	end

	return list
end

-- [Table Utilities] ********************************************************************************************************************************

function HMUtils.getTableLength(tbl)
-- Returns the number of table entries
-- Used for tables that have non-consecutive keys
	-- [tbl]: Target table

	if HMUtils.argIsInvalid(HMUtils.isTable(tbl), "tbl") then
		return
	end

	local idx = 0

	for key, val in pairs(tbl) do
		idx = idx + 1
	end

	return idx
end

function HMUtils.getTable(source, path, convert, verbose)
-- Returns a nested table based on a string path
	-- [source]:  Source table
	-- [path]:	  Table path > Optional
	-- [convert]: Convert numeric strings to numbers > Default = true
	-- [verbose]: Show extra errors/messages > Default = false

	path 	= Utils.getNoNil(path, "")
	convert = Utils.getNoNil(convert, true)
	verbose = Utils.getNoNil(verbose, false)

	if HMUtils.argIsInvalid(HMUtils.isTable(source), "source")
	or HMUtils.argIsInvalid(HMUtils.isString(path), "path")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(verbose), "verbose")
	-- convert handed by HMUtils.splitString()
	then
		return
	end

	if path ~= "" then
		local entries = HMUtils.splitString(path, ".", convert)
		for _, entry in ipairs(entries) do
			source = source[entry]
			if not HMUtils.isTable(source) then
				if not HMUtils.errorOnTrue(verbose == true, "INVALID_TABLE_NAME", path) then
					HMUtils.debugMsg(HMUtils.DEBUG.TESTING, "INVALID_TABLE_NAME", path)
				end
				return
			end
		end
	end

	return source
end

function HMUtils.createTable(source, path, convert)
-- Creates and returns a nested table based on a string path
	-- [source]:  Source table
	-- [path]:	  Table path
	-- [convert]: Convert numeric strings to numbers > Default = true

	if HMUtils.argIsInvalid(HMUtils.isTable(source), "source")
	or HMUtils.argIsInvalid(HMUtils.isString(path), "path")
	-- convert handled by HMUtils.splitString()
	then
		return
	end

	local entries = HMUtils.splitString(path, ".", convert)
	for _, entry in ipairs(entries) do
		source[entry] = Utils.getNoNil(source[entry], {})
		source = source[entry]
	end

	return source
end

function HMUtils.pack(...)
-- Returns a table containing all values (including nil values) as well as a "n" field for number of entries
	-- [...]: Values to pack (any)

	return {n = select("#", ...), ...}
end

function HMUtils.unpack(tbl, idx)
-- Returns a list of values from a table, starting with index
	-- [tbl]: Source table
	-- [idx]: Index to start > Default = 1

	idx = Utils.getNoNil(idx, 1)

	if HMUtils.argIsInvalid(HMUtils.isTable(tbl), "tbl")
	or HMUtils.argIsInvalid(HMUtils.isNumber(idx), "idx")
	then
		return
	end

	local numEntries = Utils.getNoNil(tbl.n, HMUtils.getTableLength(tbl))
	if numEntries > 0 then
		if idx < 1 or idx > numEntries then
			HMUtils.errorMsg("INVALID ARGUMENT", "idx")
			return
		end

		return unpack(tbl, idx, numEntries)
	end
end

function HMUtils.sortTable(tbl, reverse)
-- Returns a table sorted in ascending or descending order
	-- [tbl]: 	  Table to sort
	-- [reverse]: Sort backwards > Default = false

	reverse = Utils.getNoNil(reverse, false)

	if HMUtils.argIsInvalid(HMUtils.isTable(tbl), "tbl")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(reverse), "reverse")
	then
		return
	end

	local tblNew = {}
	for k, v in pairs(tbl) do
		table.insert(tblNew, k)
	end

	if reverse == true then
		table.sort(tblNew, function(v1, v2) return v1 > v2 end)
	else
		table.sort(tblNew, function(v1, v2) return v1 < v2 end)
	end

	return tblNew
end

function HMUtils.sortTableByKey(tbl, key, reverse, allowStrings)
-- Returns a table sorted by the specified key
	-- [tbl]: 	  	  	Table to sort
	-- [key]: 	  	  	Key to use for sorting
	-- [reverse]: 	  	Reverse sort order > Default = false
	-- [allowStrings]:	Allow sorting of string values > Default = true

	reverse = Utils.getNoNil(reverse, false)
	allowStrings = Utils.getNoNil(allowStrings, true)

	if HMUtils.argIsInvalid(HMUtils.isTable(tbl), "tbl")
	or HMUtils.argIsInvalid(HMUtils.isString(key), "key")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(reverse), "reverse")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(allowStrings), "allowStrings")
	then
		return
	end

	local iterFunc = pairs
	if HMUtils.getTableLength(tbl) == #tbl then
		iterFunc = ipairs
	end

	local tblNew = {}
	for k, v in iterFunc(tbl) do
		if HMUtils.errorOnFalse(HMUtils.isTable(v), "INVALID_TABLE_NAME", string.format("key %q not found", key))
		or HMUtils.errorOnFalse(HMUtils.isNumber(v[key]) or HMUtils.isString(v[key]), "INVALID_TABLE_NAME", string.format("key %q is invalid", key))
		or HMUtils.errorOnTrue(allowStrings == false and HMUtils.isString(v[key]), "INVALID_TABLE_NAME", string.format("strings not allowed for key %q", key))
		then
			return
		end
		table.insert(tblNew, v)
	end

	if reverse == true then
		table.sort(tblNew, function(val1, val2) return val1[key] > val2[key] end)
	else
		table.sort(tblNew, function(val1, val2) return val1[key] < val2[key] end)
	end

	return tblNew
end

function HMUtils.indexTableByKey(tbl, key)
-- Returns an array indexed by the specified key
-- Syncs original table values to match new index
	-- [tbl]: 	  Table to sort
	-- [key]: 	  Key to use for sorting

	-- Errors handled by HMUtils.sortTableByKey()

	local tblNew = HMUtils.sortTableByKey(tbl, key, false, false)
	if not tblNew then
		return
	end

	-- Fix duplicate entries
	for i, v in ipairs(tblNew) do
		v[key] = i
	end

	return tblNew
end

-- [Function Utilities] *****************************************************************************************************************************

function HMUtils.argIsInvalid(expression, name, debugOnly)
-- Evaluates and expression, displaying an invalid argument error on failure
-- Shortcut function for checking function arguments
	-- [expression]:	Expression to check
	-- [name]:			Argument name
	-- [debugOnly]:		Display a debug message instead of an error > Default = false

	debugOnly = Utils.getNoNil(debugOnly, false)

	if HMUtils.isNumber(name) then
		name = tostring(name)
	end

	if HMUtils.errorOnFalse(HMUtils.isString(name), "INVALID_ARGUMENT", "name")
	or HMUtils.errorOnFalse(HMUtils.isBoolean(debugOnly), "INVALID_ARGUMENT", "debugOnly")
	then
		return
	end

	if not expression then
		if debugOnly then
			if HMUtils.debugMsg(HMUtils.DEBUG.ERROR, "INVALID_ARGUMENT", name) then
				printCallstack()
			end
		else
			HMUtils.errorMsg("INVALID_ARGUMENT", name)
		end
		return true
	end
end

function HMUtils.getRealArg(args, index)
-- Returns function argument offset by number of function (superFunc) values >> Not really sure if this is necessary
	-- [args]:	Arguments to check
	-- [index]:	Index of value to get

	if HMUtils.argIsInvalid(HMUtils.isTable(args), "args")
	or HMUtils.argIsInvalid(HMUtils.isNumber(index) and index > 0, "index")
	then
		return
	end

	for _, arg in ipairs(args) do
		if not HMUtils.isFunction(arg) then
			break
		end
		index = index + 1
	end

	return args[index]
end

function HMUtils.setRealArg(args, index, value)
-- Sets argument value offset by number of function (superFunc) values >> Not really sure if this is necessary
	-- [args]:	Arguments table
	-- [index]:	Index of value to set
	-- [value]: Value to set

	if HMUtils.argIsInvalid(HMUtils.isTable(args), "args")
	or HMUtils.argIsInvalid(HMUtils.isNumber(index) and index > 0, "index")
	then
		return
	end

	for _, arg in ipairs(args) do
		if not HMUtils.isFunction(arg) then
			break
		end
		index = index + 1
	end

	args[index] = value
end

function HMUtils.getFunctionName(path, convert)
-- Returns a function name and containing table based on the given path
	-- [path]:	  Function path
	-- [convert]: Convert numeric strings to numbers > Default = true

	convert = Utils.getNoNil(convert, true)

	if HMUtils.argIsInvalid(HMUtils.isString(path), "path")
	-- convert handled by HMUtils.splitString()
	then
		return
	end

	local entries = HMUtils.splitString(path, ".", convert)
	local funcName = nil
	local funcPath = nil

	for i, entry in ipairs(entries) do
		if i == #entries then
			funcName = entry
		elseif i == 1 then
			funcPath = entry
		else
			funcPath = funcPath.."."..entry
		end
	end

	return funcName, funcPath
end

function HMUtils.getFunction(source, path, verbose)
-- Returns a function based on the given string path
	-- [source]:	Source function or table
	-- [path]:		Path to function > Only used when source is a table
	-- [verbose]:	Show extra errors/messages > Default = false

	verbose = Utils.getNoNil(verbose, false)

	if HMUtils.argIsInvalid(HMUtils.isTable(source) or HMUtils.isFunction(source), "source")
	or HMUtils.argIsInvalid(path == nil or HMUtils.isString(path), "path")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(verbose), "verbose")
	then
		return
	end

	if HMUtils.isFunction(source) then
		return source
	else
		if not path then
			HMUtils.errorMsg("INVALID_ARGUMENT", "path")
			return
		end

		local funcName, funcPath = HMUtils.getFunctionName(path)

		if funcPath then
			source = HMUtils.getTable(source, funcPath)
			if not source then
				local errMsg = "INVALID_TABLE_NAME"
				if not HMUtils.errorOnTrue(verbose, errMsg, funcPath) then
					HMUtils.debugMsg(HMUtils.DEBUG.FULL, errMsg, funcPath)
				end
				return
			end
		end

		local retFunc = source[funcName]
		if not HMUtils.isFunction(retFunc) then
			local errMsg = "INVALID_FUNCTION_NAME"
			if not HMUtils.errorOnTrue(verbose, errMsg, funcName) then
				HMUtils.debugMsg(HMUtils.DEBUG.FULL, errMsg, funcName)
			end
			return
		end

		return retFunc
	end
end

function HMUtils.getSavedFunction(app, path, source, verbose)
-- Returns the saved version of an overwritten function
	-- [app]:	 	Source application
	-- [path]:		Path to saved function
	-- [source]: 	Source table
	-- [verbose]:   Show extra errors/messages > Default = true

	verbose = Utils.getNoNil(verbose, true)

	if HMUtils.argIsInvalid(HMUtils.isApp(app), "app")
	or HMUtils.argIsInvalid(HMUtils.isString(path), "path")
	or HMUtils.argIsInvalid(source == nil or HMUtils.isTable(source), "source")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(verbose), "verbose")
	then
		return
	end

	local savePath = "savedFunctions."..path

	if source then
		savePath = app.name.."."..savePath
	else
		source = app
	end

	local retFunc = HMUtils.getFunction(source, savePath)
	if not retFunc then
		local errMsg = "INVALID_SAVED_FUNCTION"
		if not HMUtils.errorOnTrue(verbose, errMsg, savePath) then
			HMUtils.debugMsg(HMUtils.DEBUG.FULL, errMsg, savePath)
		end
		return
	end

	return retFunc
end

function HMUtils.saveFunction(app, func, path, source, overwrite, verbose)
-- Saves and returns a function
	-- [app]:		Source application
	-- [func]:		Function to save
	-- [path]: 		Path to function or function name
	-- [source]:	Source table > Optional
	-- [overwrite]:	Overwrite currently saved function > Default = false
	-- [verbose]: 	Show extra errors/messages > Default = true

	overwrite = Utils.getNoNil(overwrite, false)
	verbose = Utils.getNoNil(verbose, true)

	if HMUtils.argIsInvalid(HMUtils.isApp(app), "app")
	or HMUtils.argIsInvalid(HMUtils.isFunction(func), "func")
	or HMUtils.argIsInvalid(HMUtils.isString(path), "path")
	or HMUtils.argIsInvalid(source == nil or HMUtils.isTable(source), "source")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(overwrite), "overwrite")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(verbose), "verbose")
	then
		return
	end

	local savePath = "savedFunctions"
	local funcName, funcPath = HMUtils.getFunctionName(path)

	if funcPath then
		savePath = savePath.."."..funcPath
	end

	if source then
		source = HMUtils.createTable(source, app.name.."."..savePath)
	else
		source = HMUtils.createTable(app, savePath)
	end

	if HMUtils.getFunction(source, funcName) and not overwrite then
		local errMsg = "Saved function %q already exists"
		if not HMUtils.errorOnTrue(verbose, errMsg, funcName) then
			HMUtils.debugMsg(HMUtils.DEBUG.FULL, errMsg, funcName)
		end
		return false -- send false instead of nil on overwrite errors
	end

	source[funcName] = func
	return source[funcName]
end

function HMUtils.setFunction(app, className, path, required, target)
-- Saves, overwrites, and returns a function
-- Overwrites if original function is present, creates a blank original function otherwise (used for future-proofing)
	-- [app]:		Source application
	-- [className]:	Source class name > Optional
	-- [path]:		Path to original and saved function
	-- [required]: 	Original function is required > Default = true
	-- [target]:	Target table for saved function > Optional

	required = Utils.getNoNil(required, true)

	if HMUtils.argIsInvalid(HMUtils.isApp(app), "app")
	or HMUtils.argIsInvalid(className == nil or HMUtils.isTable(_G[className]), "className")
	or HMUtils.argIsInvalid(HMUtils.isString(path), "path")
	or HMUtils.argIsInvalid(HMUtils.isBoolean(required), "required")
	or HMUtils.argIsInvalid(target == nil or HMUtils.isTable(target), "target")
	then
		return
	end

	local source = Utils.getNoNil(target, Utils.getNoNil(_G[className], _G))
	local funcName, funcPath = HMUtils.getFunctionName(path)
	local savePath = path

	if className then
		savePath = className.."."..savePath
	end

	local oldFunction = HMUtils.getFunction(source, path)
	local newFunction = HMUtils.getFunction(app:class(), savePath)

	if not oldFunction then
		if not required then
			HMUtils.debugMsg(HMUtils.DEBUG.FULL, "INVALID_OLD_FUNCTION_NAME", funcName)
			oldFunction = function(...) end
			if funcPath then
				source = HMUtils.createTable(source, funcPath)
			end
		end
	else
		if funcPath then
			source = HMUtils.getTable(source, funcPath)
		end
	end

	if HMUtils.errorOnFalse(HMUtils.isFunction(oldFunction), "INVALID_OLD_FUNCTION_NAME", funcName)
	or HMUtils.errorOnFalse(HMUtils.isFunction(newFunction), "INVALID_NEW_FUNCTION_NAME", funcName)
	or not HMUtils.saveFunction(app, oldFunction, savePath, target) -- errors handled by HMUtils.saveFunction()
	then
		return
	end

	source[funcName] = newFunction
	return source[funcName]
end

-- [FS19 General Utilities] *************************************************************************************************************************

function HMUtils.getText(key, ...)
-- Wrapper for g_i18n:getText with string formatting features built in
-- Returns the formatted g_i18n text (or nil if not found)
	-- [key]: Text key to retrieve (number, string)
	-- [...]: Variables to pass to the built-in string.format function (any)

	if HMUtils.argIsInvalid(HMUtils.isString(key), "key") then
		return
	end

	return string.format(g_i18n:getText(key), ...)
end

-- [FS19 Fill Type Utilities] ***********************************************************************************************************************

function HMUtils.getFillType(fillType)
-- Returns the specified fill type table
	-- [fillType]: Fill type to return (index or name)

	if HMUtils.isNumber(fillType) then
		fillType = g_fillTypeManager:getFillTypeByIndex(fillType)
	elseif HMUtils.isString(fillType) then
		fillType = g_fillTypeManager:getFillTypeByName(fillType)
	else
		return
	end

	return fillType
end

-- [FS19 Fruit Type Utilities] **********************************************************************************************************************

function HMUtils.getFruitType(fruitType)
-- Returns the specified fruit type table
	-- [fruitType]: Fruit type to return (index or name)

	if HMUtils.isNumber(fruitType) then
		fruitType = g_fruitTypeManager:getFruitTypeByIndex(fruitType)
	elseif HMUtils.isString(fruitType) then
		fruitType = g_fruitTypeManager:getFruitTypeByName(fruitType)
	else
		return
	end

	return fruitType
end
